﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System;
using Microsoft.Xna.Framework;
using GE.World.Entities;

namespace GE.Physics.Shapes
{
    class DynamicShapeRectangle : ShapeRectangle
    {
        #region Variables"

        /// <summary>
        /// axis aligned bounding box
        /// </summary>
        private Vector2[] _v2BoundingBox;//[4];

        /// <summary>
        /// oriented bounding box
        /// </summary>
        private Vector2[] _v2Obb;//[4];

        #endregion

        /// <summary>
        /// Constructor. Set the id to -1.
        /// </summary>
        /// <param name="width">Shape's width</param>
        /// <param name="height">Shape's height</param>
        /// <param name="rotationCenter">Point used as rotation center (origin used for the shape)</param>
        /// <param name="owner">Entity owner of the shape</param>
        public DynamicShapeRectangle(int width, int height, Vector2 rotationCenter, WorldEntity owner) 
            :base(width, height, rotationCenter, owner)
        {
            _iId = -1;

            //position of the geometric center of the rectangle
            _v2position = new Vector2(0);
            _fOrientation = 0;

            //m_state
            _bCollisionEnable = true;

            _v2BoundingBox = new Vector2[4];

            _v2Obb = new Vector2[4];
            _v2Obb[0] = new Vector2(0, 0);
            _v2Obb[1] = new Vector2(0, 0);
            _v2Obb[2] = new Vector2(0, 0);
            _v2Obb[3] = new Vector2(0, 0);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="id">Id of the shape. Must be uniqe and should be generated by the Physics component.</param>
        /// <param name="width">Shape's width</param>
        /// <param name="height">Shape's height</param>
        /// <param name="rotationCenter">Point used as rotation center (origin used for the shape)</param>
        /// <param name="owner">Entity owner of the shape</param>
        public DynamicShapeRectangle(int id, int width, int height, Vector2 rotationCenter, WorldEntity owner)
            : this(width, height, rotationCenter, owner)
        {
            _iId = id; 
        }

        /// <summary>
        /// Compute the axis aligned bounding box
        /// </summary>
        public void computeBoundingBox()
        {
            _v2BoundingBox[0] = new Vector2((float)-_v2RotationCenter.X, -(float)_v2RotationCenter.Y);
            _v2BoundingBox[1] = new Vector2((float)-_v2RotationCenter.X + _iWidth, -(float)_v2RotationCenter.Y);
            _v2BoundingBox[2] = new Vector2((float)-_v2RotationCenter.X + _iWidth, -(float)_v2RotationCenter.Y + _iHeight);
            _v2BoundingBox[3] = new Vector2((float)-_v2RotationCenter.X, -(float)_v2RotationCenter.Y + _iHeight);
        }

        /// <summary>
        /// Compute and return the oriented bounding box
        /// </summary>
        /// <returns>an array of 4 vector2 representing the rectangle</returns>
        public override Vector2[] getOrientedBoundingBox()
        {
            for (int i = 0; i < 4; i++)
            {
                float cosinus = (float)Math.Cos(_fOrientation);
                float sinus = (float)Math.Sin(_fOrientation);

                _v2Obb[i].X = cosinus * _v2BoundingBox[i].X - sinus * _v2BoundingBox[i].Y + _v2position.X;

                _v2Obb[i].Y = cosinus * _v2BoundingBox[i].Y + sinus * _v2BoundingBox[i].X + _v2position.Y;
              
            }

            return _v2Obb;
        }

        /// <summary>
        /// Resize the shape
        /// </summary>
        /// <param name="width">The new width</param>
        /// <param name="height">The new height</param>
        public override void resize(int width, int height)
        {
            _iWidth = width;
            _iHeight = height;
            computeBoundingBox();
        }
    }
}
